
/*
  CLASSiC DAC, Copyright 2013 SILICON CHIP Publications
  playback.c: WAV playback control logic
  Written by Nicholas Vinen, 2010-2013
*/

#include "playback.h"
#include "sdcard/sd.h"
#include "sdcard/ff.h"
#include "wav.h"
#include "DCI.h"
#include "PLL1708.h"
#include "Timer.h"
#include "p33Fxxxx.h"
#include <string.h>
#include <stdlib.h>
#include <math.h>

//////////// State variables ////////////

unsigned char/*playback_state_t*/ playback_state;
unsigned char/*playback_error_t*/ last_playback_error;
unsigned char/*playback_order_t*/ desired_playback_order, current_playback_order;

#define INIT_ATTEMPTS 3
#define INIT_DELAY    200
#define SORT_ENTRIES     256
#define SORT_ENTRY_SIZE (DCI_BUFFER_PAGES*512/SORT_ENTRIES)
#define MAX_FOLDER_DEPTH 8

extern CARD_INFO cardInfo;
static FATFS fs;
static DIR cur_pos;
static FILINFO cur_file_info;
static FIL cur_file;
static unsigned long wav_data_start;
static unsigned long wav_bytes_remaining;
static unsigned long wav_data_size;
static unsigned short wav_data_align;
static unsigned long wav_sample_rate;
static unsigned char wav_num_channels, wav_bytes_per_sample, wav_dummy, muted;
static unsigned char transitioning_to_next_file, order_pos, play_just_one, dont_repeat_all;
static unsigned char order[SORT_ENTRIES];
static unsigned short order_num_files;

static unsigned char cur_folder_level, cur_seed_level;
static unsigned short folder_seeds[MAX_FOLDER_DEPTH];
static char cur_path[_MAX_LFN+5];

//////////// Utility functions ////////////

static int strcasecmp(const char* s1, const char* s2) {
	while(*s1 && *s2) {
		if( *s1 != *s2 )
			return (int)*s1 - (int)*s2;
		if( !*s1 )
			return 0;
		++s1;
		++s2;
	}
	return *s1 == *s2;
}

//////////// Crossfeed is implemented at the playback level, here are the required variables & functions ////////////

#define CROSSFEED_MAX_DELAY 32

unsigned char CrossFeed_Enabled, Crossfeed_Delay, Crossfeed_Atten;
signed short CrossFeed_DelayBuf[CROSSFEED_MAX_DELAY][2];
unsigned short CrossFeed_DelayBufPos;
signed short CrossFeed_FilterCoeff_B, CrossFeed_FilterCoeff_A, CrossFeed_FilterHistory[2][2];
void CrossFeed_Init(unsigned long sampling_rate, unsigned short rolloff) {
  float W = tan(3.14159265 * rolloff / sampling_rate);
  float N = 1/(1+W);
  CrossFeed_FilterCoeff_A = (W-1) * N * 32768;
  CrossFeed_FilterCoeff_B =  W    * N * 32768;
  CrossFeed_DelayBufPos = 0;
  memset(CrossFeed_DelayBuf, 0, sizeof(CrossFeed_DelayBuf));
  memset(CrossFeed_FilterHistory, 0, sizeof(CrossFeed_FilterHistory));
}

void CrossFeed16(signed short* audio_data, unsigned short num_samples) {
  while( num_samples-- ) {
    signed short Acc1 = ((signed long)audio_data[0] * (unsigned long)CrossFeed_FilterCoeff_B + (signed long)CrossFeed_FilterHistory[0][0] * (unsigned long)CrossFeed_FilterCoeff_B - (signed long)CrossFeed_FilterHistory[0][1] * (unsigned long)CrossFeed_FilterCoeff_A)>>15;
    CrossFeed_FilterHistory[0][1] = Acc1;
    CrossFeed_FilterHistory[0][0] = audio_data[0];
    audio_data[0] += (CrossFeed_DelayBuf[CrossFeed_DelayBufPos][1]>>Crossfeed_Atten) - (audio_data[0]>>Crossfeed_Atten);

    signed short Acc2 = ((signed long)audio_data[1] * (unsigned long)CrossFeed_FilterCoeff_B + (signed long)CrossFeed_FilterHistory[1][0] * (unsigned long)CrossFeed_FilterCoeff_B - (signed long)CrossFeed_FilterHistory[1][1] * (unsigned long)CrossFeed_FilterCoeff_A)>>15;
    CrossFeed_FilterHistory[1][1] = Acc2;
    CrossFeed_FilterHistory[1][0] = audio_data[1];
    audio_data[1] += (CrossFeed_DelayBuf[CrossFeed_DelayBufPos][0]>>Crossfeed_Atten) - (audio_data[1]>>Crossfeed_Atten);

    CrossFeed_DelayBuf[CrossFeed_DelayBufPos][0] = Acc1;
    CrossFeed_DelayBuf[CrossFeed_DelayBufPos][1] = Acc2;

    audio_data += 2;
    if( ++CrossFeed_DelayBufPos == Crossfeed_Delay )
      CrossFeed_DelayBufPos = 0;
  }
}

void CrossFeed24(signed short* audio_data, unsigned short num_samples) {
  while( num_samples-- ) {
    signed short Acc1 = ((signed long)audio_data[0] * (unsigned long)CrossFeed_FilterCoeff_B + (signed long)CrossFeed_FilterHistory[0][0] * (unsigned long)CrossFeed_FilterCoeff_B - (signed long)CrossFeed_FilterHistory[0][1] * (unsigned long)CrossFeed_FilterCoeff_A)>>16;
    CrossFeed_FilterHistory[0][1] = Acc1;
    CrossFeed_FilterHistory[0][0] = audio_data[0];
    audio_data[0] += (CrossFeed_DelayBuf[CrossFeed_DelayBufPos][1]>>Crossfeed_Atten) - (audio_data[0]>>Crossfeed_Atten);

    signed short right = ((unsigned char*)audio_data)[3] | (((signed short)((signed char*)audio_data)[4])<<8);
    signed short Acc2 = ((signed long)right * (unsigned long)CrossFeed_FilterCoeff_B + (signed long)CrossFeed_FilterHistory[1][0] * (unsigned long)CrossFeed_FilterCoeff_B - (signed long)CrossFeed_FilterHistory[1][1] * (unsigned long)CrossFeed_FilterCoeff_A)>>16;
    CrossFeed_FilterHistory[1][1] = Acc2;
    CrossFeed_FilterHistory[1][0] = right;
            right += (CrossFeed_DelayBuf[CrossFeed_DelayBufPos][0]>>Crossfeed_Atten) - (        right>>Crossfeed_Atten);
    ((unsigned char*)audio_data)[3] = right;
    ((signed char*)audio_data)[4] = right>>8;

    CrossFeed_DelayBuf[CrossFeed_DelayBufPos][0] = Acc1;
    CrossFeed_DelayBuf[CrossFeed_DelayBufPos][1] = Acc2;

    audio_data += 3;
    if( ++CrossFeed_DelayBufPos == Crossfeed_Delay )
      CrossFeed_DelayBufPos = 0;
  }
}

unsigned long g_current_sample_rate;
unsigned char Playback_Set_Crossfeed(bool enabled, unsigned short FilterFreq, unsigned char Delay, unsigned char Atten) {
  if( enabled ) {
    memset(CrossFeed_FilterHistory, 0, sizeof(CrossFeed_FilterHistory));
    CrossFeed_Init(g_current_sample_rate, FilterFreq);
    if( Delay > CROSSFEED_MAX_DELAY )
      Delay = CROSSFEED_MAX_DELAY;
    Crossfeed_Delay = Delay;
    Crossfeed_Atten = Atten;
  }
  CrossFeed_Enabled = enabled;
  return last_playback_error;
}

unsigned char Playback_Get_Crossfeed_Status() {
  return CrossFeed_Enabled;
}

//////////// Functions below mainly deal with opening WAV files, scanning directories and figuring out which file to play next ////////////

static inline bool set_sample_rate(unsigned long rate) {
  if( rate ) {
    if( PLL1708_Configure(rate) ) {
      g_current_sample_rate = rate;
      return true;
    } else {
      return false;
    }
  } else {
    return false;
  }
}

#ifdef __DEBUG
extern void playback_error();
#endif

unsigned char do_open_wav() {
	char wavbuf[512];
	unsigned int nRead;
	WAVHeader header;

	if( f_read(&cur_file, (BYTE*)wavbuf, sizeof(wavbuf), &nRead) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = file_read_error;
		return last_playback_error;
	}
	WAVinitHeader(&header);
	if( !WAVreadWAVHeader((BYTE*)wavbuf, &header, nRead) ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = invalid_wav_file;
		return last_playback_error;
	}
	if( header.fmtHeader.audioFormat != 1 || (header.fmtHeader.numChannels != 1 && header.fmtHeader.numChannels != 2) ||
       (header.fmtHeader.bitsPerSample != 16 && header.fmtHeader.bitsPerSample != 24) || !set_sample_rate(header.fmtHeader.sampleRate) ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = invalid_wav_format;
		return last_playback_error;
	}
	wav_data_start = header.dataHeader.dataPtr - (BYTE*)wavbuf;
	if( f_lseek(&cur_file, wav_data_start) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = invalid_wav_file;
		return last_playback_error;
	}
	wav_data_size = wav_bytes_remaining = header.dataHeader.data.chunkSize;
	wav_data_align = 512 - ((512 - (wav_data_start&511)) & ~(header.fmtHeader.numChannels * header.fmtHeader.bitsPerSample / 8 - 1));
	if( header.fmtHeader.numChannels == 1 && wav_data_align > 256 )
		wav_data_align -= 256;
	wav_sample_rate = header.fmtHeader.sampleRate;
	wav_num_channels = header.fmtHeader.numChannels;
	wav_bytes_per_sample = header.fmtHeader.numChannels * header.fmtHeader.bitsPerSample / 8;
	wav_dummy = 0;

    if( DCI_BitDepth != (wav_bytes_per_sample<<2) ) {
      DCI_BitDepth = wav_bytes_per_sample<<2;
      DCI_Reset_Buffer();
    } else if( DCI_BitDepth == 24 ) {
      while( !DCI_Write_Will_Be_on_24bit_Boundary() ) {
        memset(wavbuf, 0, sizeof(wavbuf));
        DCI_Write((const signed short*)wavbuf, 2);
      }
    }

	return 0;
}

static unsigned char open_folder() {
	int len = strlen(cur_path);
	int flen = strlen(cur_file_info.fname);
	if( len + flen + (len > 3 ? 1 : 0) < sizeof(cur_path) ) {
        if( len > 3 )
            cur_path[len++] = '\\';
		strcpy(cur_path+len, cur_file_info.fname);
		++cur_folder_level;
		current_playback_order = directory;
	} else {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = max_depth_exceeded;
	}
	return last_playback_error;
}

static unsigned char update_playback_order(unsigned char retain_old_pos);
static unsigned char goto_last_wav_file();
static unsigned char goto_last_dir_or_first_wav_file();

unsigned char open_current_file() {
	char path[_MAX_LFN+5];
   	int len = strlen(cur_path);
	strcpy(path, cur_path);
	if( path[len-1] != '\\' )
		path[len++] = '\\';
	strncpy(path+len, cur_file_info.fname, sizeof(path)-len);
    if( strlen(cur_file_info.fname) < 12 ) {
      len += strlen(cur_file_info.fname);
	  strncpy(path+len, cur_file_info.fname+8, sizeof(path)-len);
    }
	path[sizeof(path)-1] = '\0';
	return f_open(&cur_file, path, FA_READ|FA_OPEN_EXISTING) == FR_OK;
}

static unsigned char open_wav_file(bool bReverse, bool bReverseDir) {
	if( (cur_file_info.fattrib&AM_DIR) ) {
		if( open_folder() != ok )
			return last_playback_error;
		if( bReverse )
			return goto_last_wav_file();
        if( bReverseDir )
			return goto_last_dir_or_first_wav_file();
		else
			return Playback_Go_To_First_file();
	} else {
		if( !open_current_file() ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( do_open_wav() )
			return last_playback_error;

		playback_state = stopped;
		last_playback_error = ok;
		return last_playback_error;
	}
}

static void insert_shuffle_index(unsigned short pos, unsigned short num, unsigned char which) {
	while( pos <= num ) {
		unsigned char old = order[pos];
		order[pos] = which;
		which = old;
		++pos;
	}
}

unsigned long gerhard_random;
unsigned short get_gerhard_random() {
	gerhard_random = (gerhard_random * 32719 + 3) % 32749;
	return gerhard_random;
}
void seed_gerhard_random(unsigned short seed) {
	gerhard_random = seed + (((unsigned long)seed)<<16);
	get_gerhard_random();
	gerhard_random += seed;
}

static unsigned char goto_file_index(unsigned short index) {
	const char* p;
	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( ((cur_file_info.fattrib&AM_DIR) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			if( index-- == 0 )
				return last_playback_error;
		}
	} while( cur_pos.sect != 0 );

	// this should not happen as we should only be passing in the index of files that exist
#ifdef __DEBUG
	playback_error();
#endif
	playback_state = error;
	last_playback_error = no_wav_files;
	return last_playback_error;
}

static int goto_file_name(const char* name, unsigned char search_long_names) {
	const char* p;
    unsigned short index = 0;
	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return -1;
	}
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return -1;
		}
		if( ((cur_file_info.fattrib&AM_DIR) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			if( !strcmp((const char*)cur_file_info.fname, (const char*)name) || (search_long_names && !strcasecmp((const char*)cur_file_info.lfname, (const char*)name)) )
				return index;
			else
				++index;
		}
	} while( cur_pos.sect != 0 );

	// this should not happen as we should only be passing in the index of files that exist
#ifdef __DEBUG
	playback_error();
#endif
	playback_state = error;
	last_playback_error = no_wav_files;
	return -1;
}

static unsigned char find_file_index(unsigned char file_index) {
	unsigned short i;
	for( i = 0; i < order_num_files; ++i )
		if( order[i] == file_index )
			return i;
	return 0;
}

static unsigned short GetRandomSeed() {
  return TMR4;
}

static unsigned char init_shuffle(unsigned char retain_old_pos) {
	unsigned short i;
	FILINFO old_file_info;
	unsigned short old_file_index = 0;
	const char* p;

	order_num_files = 0;
	memcpy(&old_file_info, &cur_file_info, sizeof(old_file_info));

	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			if( !strcmp(cur_file_info.fname, old_file_info.fname) )
				old_file_index = order_num_files;
			if( ++order_num_files == SORT_ENTRIES )
				break;
		}
	} while( cur_pos.sect != 0 );

	if( cur_folder_level >= cur_seed_level )
		folder_seeds[cur_folder_level] = GetRandomSeed();
	cur_seed_level = cur_folder_level+1;
	seed_gerhard_random(folder_seeds[cur_folder_level]);

	for( i = 0; i < order_num_files; ++i ) {
		insert_shuffle_index(get_gerhard_random()%(i+1), i, i);
	}

	if( order_num_files == 0 ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = no_wav_files;
	} else {
		order_pos = retain_old_pos ? find_file_index(old_file_index) : 0;
	}

	if( playback_state == stopped )
		return goto_file_index(order[order_pos]);
	else
		return last_playback_error;
}

typedef int (*compfunc_t) (const void*, const void*);

static unsigned char init_sort(unsigned char retain_old_pos) {
	unsigned short i;
	unsigned char* sort_buffer;
	unsigned char* dest;
	const char* p;
	FILINFO old_file_info;
	unsigned short old_file_index = 0;

	order_num_files = 0;
	memcpy(&old_file_info, &cur_file_info, sizeof(old_file_info));

	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	dest = sort_buffer = DCI_Get_Buffer(); // DCI isn't running while we're doing this so we can use its buffer to sort the file names
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			if( !strcmp(cur_file_info.fname, old_file_info.fname) )
				old_file_index = order_num_files;
			strncpy((char*)dest, cur_file_info.lfname, SORT_ENTRY_SIZE-2);
			dest[SORT_ENTRY_SIZE-2] = '\0';
			dest[SORT_ENTRY_SIZE-1] = order_num_files;
			dest += SORT_ENTRY_SIZE;
			if( ++order_num_files == SORT_ENTRIES )
				break;
		}
	} while( cur_pos.sect != 0 );

	qsort(sort_buffer, order_num_files, SORT_ENTRY_SIZE, (compfunc_t)&strcmp);

	dest = sort_buffer+SORT_ENTRY_SIZE-1;
	for( i = 0; i < order_num_files; ++i ) {
		order[i] = dest[0];
		dest += SORT_ENTRY_SIZE;
	}

	memset(sort_buffer, 0, SORT_ENTRY_SIZE*SORT_ENTRIES);

	if( order_num_files == 0 ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = no_wav_files;
	} else {
		order_pos = retain_old_pos ? find_file_index(old_file_index) : 0;
	}

	if( playback_state == stopped )
		return goto_file_index(order[order_pos]);
	else
		return last_playback_error;
}

static unsigned char update_playback_order(unsigned char retain_old_pos) {
	if( desired_playback_order != current_playback_order ) {
		current_playback_order = desired_playback_order;
		if( desired_playback_order == shuffle )
			return init_shuffle(retain_old_pos);
		else if( desired_playback_order == sorted )
			return init_sort(retain_old_pos);
	}
	return last_playback_error;
}

static unsigned char goto_next_wav_file(unsigned char retain_old_pos) {
	unsigned char found = 0;
	unsigned char loops = 0;

	if( update_playback_order(retain_old_pos) != ok )
		return last_playback_error;
	if( !retain_old_pos )
		order_pos = SORT_ENTRIES-1;

try_next_file:
	if( current_playback_order != directory ) {
		if( ++order_pos >= order_num_files ) {
			if( cur_folder_level > 0 ) {
				Playback_Up_Folder();
				return goto_next_wav_file(true);
			} else {
				order_pos = 0;
				if( ++loops > 1 ) {
					playback_state = error;
					last_playback_error = no_wav_files;
					return last_playback_error;
				}
			}
		}
		if( goto_file_index(order[order_pos]) )
			return last_playback_error;
		else
			goto try_open;
	}

	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			found = 1;
			break;
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		if( cur_folder_level > 0 && Playback_Up_Folder() == ok )
			return goto_next_wav_file(retain_old_pos);
		else
			return Playback_Go_To_First_file();
	} else {
		unsigned char init_folder_level, init_playback_state;
	try_open:
		init_folder_level = cur_folder_level;
		init_playback_state = playback_state;
		if( open_wav_file(false, false) == no_wav_files && (current_playback_order != directory || order_pos+1 < order_num_files) ) {
			playback_state = init_playback_state;
			last_playback_error = ok;
			while( cur_folder_level > init_folder_level )
				if( Playback_Up_Folder() != ok )
					return last_playback_error;
			if( update_playback_order(retain_old_pos) != ok )
				return last_playback_error;
			goto try_next_file;
		}
		return last_playback_error;
	}
}

unsigned char Playback_Go_To_First_file() {
	if( current_playback_order == directory && f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	return goto_next_wav_file(0);
}

unsigned char find_last_file() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_lfname[_MAX_LFN+1];
	bool found = false;

	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return 0;
	}
	strcpy(old_lfname, cur_file_info.lfname);
	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			found = true;
			memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
			memcpy(&last_pos, &cur_pos, sizeof(last_pos));
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = no_wav_files;
		return 0;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		return 1;
	}
}

unsigned char find_last_dir() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_lfname[_MAX_LFN+1];
	bool found = false;

	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return 0;
	}
	strcpy(old_lfname, cur_file_info.lfname);
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( ((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0' ) {
			found = true;
			memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
			memcpy(&last_pos, &cur_pos, sizeof(last_pos));
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		return 0;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		return 1;
	}
}

unsigned char find_prev_dir() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_lfname[_MAX_LFN+1], init_fname[13];
	bool found = false;

	memcpy(&init_fname, &cur_file_info.fname, sizeof(init_fname));
	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return 0;
	}
	strcpy(old_lfname, cur_file_info.lfname);
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( !memcmp(cur_file_info.fname, init_fname, sizeof(init_fname)) )
			break;
		if( ((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0' ) {
			found = true;
			memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
			memcpy(&last_pos, &cur_pos, sizeof(last_pos));
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		return 0;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		return 1;
	}
}

unsigned char goto_prev_wav_file(bool bGotoFirstInDir);

static unsigned char goto_last_wav_file() {
	unsigned char init_folder_level;

	if( update_playback_order(true) )
		return last_playback_error;
	if( current_playback_order == directory ) {
		if( !find_last_file() )
			return last_playback_error;
	} else {
        order_pos = order_num_files-1;
		if( goto_file_index(order[order_pos]) != ok )
			return last_playback_error;
	}

	init_folder_level = cur_folder_level;
	if( open_wav_file(true, true) == no_wav_files && (current_playback_order != directory || order_pos > 0) ) {
		while( cur_folder_level > init_folder_level )
			if( Playback_Up_Folder() != ok )
				return last_playback_error;
		if( update_playback_order(true) != ok )
			return last_playback_error;
		return goto_prev_wav_file(0);
    }      
	return last_playback_error;
}

static unsigned char goto_last_dir_or_first_wav_file() {
	if( update_playback_order(true) )
		return last_playback_error;
	if( current_playback_order == directory ) {
		if( find_last_dir() ) {
			unsigned char init_folder_level = cur_folder_level;
			while( open_wav_file(false, true) == no_wav_files ) {
				while( cur_folder_level > init_folder_level )
					if( Playback_Up_Folder() != ok )
						return last_playback_error;
				if( update_playback_order(true) != ok )
					return last_playback_error;
				if( find_prev_dir() != ok )
					return last_playback_error;
			}
		}
		if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = open_root_dir_failed;
			return last_playback_error;
		}
		return goto_next_wav_file(0);
	} else {
        for( order_pos = order_num_files-1; order_pos < order_num_files; --order_pos ) {
			if( goto_file_index(order[order_pos]) != ok )
				return last_playback_error;
			if( (cur_file_info.fattrib&AM_DIR) ) {
				if( open_wav_file(false, true) != no_wav_files )
					return last_playback_error;
			}
		}
		return Playback_Go_To_First_file();
	}
}

unsigned char find_prev_file() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_fname[13];
	char old_lfname[_MAX_LFN+1];
	unsigned char found = 0;

	strcpy(old_fname, cur_file_info.fname);
	strcpy(old_lfname, cur_file_info.lfname);
	if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return 0;
	}

	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return 0;
		}
		if( (((cur_file_info.fattrib&AM_DIR)) && cur_file_info.fname[0] != '\0') || (!(cur_file_info.fattrib&AM_DIR) && (p = strrchr(cur_file_info.fname, '.')) && is_wav_ext(p)) ) {
			if( !strcmp(cur_file_info.fname, old_fname) ) {
				break;
			} else {
				found = 1;
				memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
				memcpy(&last_pos, &cur_pos, sizeof(last_pos));
			}
		}
	} while( cur_pos.sect != 0 );

	if( !found || cur_pos.sect == 0 ) {
		return 0;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		strcpy(cur_file_info.lfname, old_lfname);
		return 1;
	}
}

unsigned char goto_prev_wav_file(bool bGotoFirstInDir) {
	unsigned char init_folder_level;

try_prev_file:
	if( update_playback_order(true) != ok )
		return last_playback_error;

	if( current_playback_order != directory ) {
		if( --order_pos >= order_num_files ) {
			if( cur_folder_level > 0 ) {
				Playback_Up_Folder();
				return goto_prev_wav_file(bGotoFirstInDir);
			} else {
				order_pos = order_num_files-1;
			}
		}

		if( goto_file_index(order[order_pos]) )
			return last_playback_error;
		// otherwise, fall through
	} else {
		if( !find_prev_file() ) {
			if( cur_folder_level > 0 ) {
				if( Playback_Up_Folder() != ok )
					return last_playback_error;
				goto try_prev_file;
			} else {
				return goto_last_wav_file();
			}
        }      
		// otherwise, fall through
	}
	init_folder_level = cur_folder_level;
	if( open_wav_file(!bGotoFirstInDir, true) == no_wav_files && (current_playback_order != directory || order_pos > 0) ) {
		while( cur_folder_level > init_folder_level )
			if( Playback_Up_Folder() != ok )
				return last_playback_error;
		goto try_prev_file;
	}
	return last_playback_error;
}

extern unsigned char memoryCardSystemUp;
extern void reset_config_to_default();
extern DSTATUS Stat;

unsigned char Playback_Reinit() {
	unsigned char i;
	char buf[4] = "0:\\";
	const char* pbuf;
	FATFS* pfs;

	for( i = 0; i < INIT_ATTEMPTS; ++i ) {
		Timer_Delay_ms_escape(INIT_DELAY, &memoryCardSystemUp, 0);
		Stat = STA_NOINIT;
		pbuf = buf;
		f_mount(0, &fs);
		if( auto_mount(&pbuf, &pfs, 0) == FR_OK )
			break;
	}
	if( i == INIT_ATTEMPTS ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		switch(cardInfo.ERROR) {
//		case ERROR_NOT_SDMMC_CARD:
		default:
			last_playback_error = sd_card_invalid_response;
			break;
		case ERROR_BAD_VOLTAGE_RANGE:
			last_playback_error = sd_card_wrong_voltage;
			break;
		case ERROR_SDMMC_CARD_TIMEOUT:
			last_playback_error = sd_card_timeout;
			break;
		}
		return last_playback_error;
	}
/*
	if( f_mount(0, &fs) != FR_OK ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = fat_mount_failed;
		return last_playback_error;
	}
*/
	current_playback_order = directory;
	return ok;
}

unsigned char Playback_Setup() {
	if( Playback_Reinit() != ok )
		return last_playback_error;
	play_just_one = 0;
   	return Playback_Go_To_First_file();
}

unsigned char Playback_Start() {
	if( playback_state == stopped || playback_state == paused ) {
		DCI_Pause(false);
		playback_state = playing;
	}
	play_just_one = 0;
	muted = 0;
	return last_playback_error;
}

unsigned char Playback_Reset() {
	playback_state = reset;
	last_playback_error = ok;
	DCI_Pause(true);
	play_just_one = 0;
	muted = 0;
	cur_folder_level = 0;
    cur_seed_level = 0;
	strcpy(cur_path, "0:\\");
    current_playback_order = directory;
	return last_playback_error;
}

unsigned char Playback_Stop() {
	if( playback_state == playing || playback_state == paused ) {
		if( playback_state == playing )
			DCI_Pause(true);
		if( f_lseek(&cur_file, wav_data_start) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = invalid_wav_file;
		} else {
			wav_data_align = 512 - ((512 - (wav_data_start&511)) & ~(wav_bytes_per_sample - 1));
			wav_bytes_remaining = wav_data_size;
			playback_state = stopped;
		}
	} else if( playback_state == stopped ) {
		return Playback_Go_To_First_file();
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char Playback_Pause() {
	if( playback_state == playing || playback_state == paused ) {
		DCI_Pause(DCI_Is_Running());
		playback_state = DCI_Is_Running() ? playing : paused;
	}
	return last_playback_error;
}

static unsigned char Playback_Next_Track_int(unsigned char clear_buffer, unsigned char loop) {
	if( clear_buffer )
		DCI_Clear_Buffer();
	if( playback_state != error ) {
		playback_state_t old_playback_state = playback_state;
		if( goto_next_wav_file(true) ) {
			if( last_playback_error == no_wav_files && current_playback_order == directory ) {
				if( !loop )
					return Playback_Reset();
				last_playback_error = ok;
				if( Playback_Go_To_First_file() ) {
					DCI_Pause(true);
					return last_playback_error;
				}
			} else {
				DCI_Pause(true);
				return last_playback_error;
			}
		} else if( current_playback_order != directory && order_pos == 0 && !loop ) {
			return Playback_Reset();
		}
		playback_state = old_playback_state;
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char Playback_Next_Track(unsigned char clear_buffer) {
	return Playback_Next_Track_int(clear_buffer, 1);
}

unsigned char Playback_Prev_Track(unsigned char clear_buffer) {
	if( playback_state == playing && clear_buffer )
		DCI_Clear_Buffer();
	if( playback_state != error ) {
		playback_state_t old_playback_state = playback_state;
		if( goto_prev_wav_file(false) ) {
			DCI_Pause(true);
			return last_playback_error;
		}
		playback_state = old_playback_state;
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char Playback_Forward() {
	if( playback_state == playing ) { // calculate number of samples in 10 seconds
		unsigned long seek_distance = (unsigned long)wav_sample_rate * (unsigned long)wav_bytes_per_sample * 10;
		seek_distance = (seek_distance + 256) & ~511; // round to nearest sector
		if( wav_bytes_remaining > seek_distance ) {
			wav_bytes_remaining -= seek_distance;
			if( !wav_dummy && f_lseek(&cur_file, (wav_data_start) + wav_data_size - wav_bytes_remaining) != FR_OK ) {
				DCI_Pause(true);
#ifdef __DEBUG
				playback_error();
#endif
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
		}
	}
	return last_playback_error;
}

unsigned char Playback_Back() {
	if( playback_state == playing ) { // calculate number of samples in 10 seconds
		unsigned long seek_distance = (unsigned long)wav_sample_rate * (unsigned long)wav_bytes_per_sample * 10;
		seek_distance = (seek_distance + 256) & ~511; // round to nearest sector
		if( wav_bytes_remaining < wav_data_size ) {
			wav_bytes_remaining += seek_distance;
			if( wav_bytes_remaining > wav_data_size )
				wav_bytes_remaining = wav_data_size;
			if( !wav_dummy && f_lseek(&cur_file, (wav_data_start) + wav_data_size - wav_bytes_remaining) != FR_OK ) {
				DCI_Pause(true);
#ifdef __DEBUG
				playback_error();
#endif
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
		}
	}
	return last_playback_error;
}

/* not used in this program...
unsigned char playback_play_single_file(unsigned char index) {
	if( update_playback_order(true) )
		return last_playback_error;

	if( playback_state == playing )
		pause_dac(true);

	if( current_playback_order == directory ) {
		unsigned char found = 0;

		if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
			playback_state = error;
			last_playback_error = open_root_dir_failed;
			return last_playback_error;
		}

		do {
			const char* p;
			if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			p = strrchr(cur_file_info.fname, '.');
			if( p && is_playable_ext(p) ) {
				if( index-- == 0 ) {
					found = 1;
					break;
				}
			}
		} while( cur_pos.sect != 0 );

		if( !found ) {
			playback_state = error;
			last_playback_error = no_wav_files;
			return last_playback_error;
		} else {
			if( open_wav_file(false, false) )
				return last_playback_error;
		}
	} else {
		if( index >= order_num_files ) {
			if( playback_state != stopped ) {
				playback_stop();
			}
			return last_playback_error;
		}
		order_pos = index-1;
		if( goto_next_wav_file(true) ) {
			if( last_playback_error == no_wav_files && current_playback_order == directory ) {
				last_playback_error = ok;
				if( Playback_Go_To_First_file() ) {
					reset_sequencer(0);
					return last_playback_error;
				}
			} else {
				return last_playback_error;
			}
		}
	}

	if( playback_state == playing ) {
		pause_dac(true);
		return last_playback_error;
	} else {
		return playback_start();
	}
}
*/

playback_state_t Playback_Get_Status() {
	return playback_state;
}

playback_error_t Playback_Get_Error() {
	return last_playback_error;
}

playback_order_t Playback_Get_Order() {
	return desired_playback_order;
}

unsigned char Playback_Set_Order(playback_order_t order) {
	desired_playback_order = order;
	return last_playback_error;
}

#define LARGE_READS

unsigned char play_file_indirect() {
	do {
		signed short wavbuf[256];
		unsigned short read_size = sizeof(wavbuf);
		unsigned char num_channels;
		UINT read = 0;
		if( transitioning_to_next_file ) {
			memset(wavbuf, 0, sizeof(wavbuf));
			num_channels = 2;
		} else {
			num_channels = wav_num_channels;
			if( num_channels == 1 )
				read_size >>= 1;
			if( wav_data_align ) {
				if( wav_data_align + read_size >= 512 )
					read_size = 512 - wav_data_align;
				memset(wavbuf, 0, wav_data_align);
			}
			if( wav_dummy ) {
				memset(wavbuf, 0, read_size);
				read = read_size;
			} else if( f_read(&cur_file, ((BYTE*)wavbuf) + wav_data_align, read_size, &read) != FR_OK ) {
				DCI_Pause(true);
#ifdef __DEBUG
				playback_error();
#endif
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			if( read > wav_bytes_remaining )
				read = wav_bytes_remaining;
			wav_bytes_remaining -= read;

            if( CrossFeed_Enabled && g_current_sample_rate <= 48000 ) {
              if( wav_bytes_per_sample == 4 )
                CrossFeed16( (signed short*)(((BYTE*)wavbuf) + wav_data_align), read>>2);
              else
                CrossFeed24( (signed short*)(((BYTE*)wavbuf) + wav_data_align), read/6);
            }

			if( wav_data_align ) {
				wav_data_align += read;
				if( wav_data_align >= 512 )
					wav_data_align = 0;
			}
			if( read != read_size || wav_bytes_remaining == 0 ) {
				memset(((BYTE*)wavbuf)+read, 0, read_size-read);
				transitioning_to_next_file = DCI_BUFFER_PAGES;
			}
		}
		DCI_Write(wavbuf, num_channels);
	} while( !DCIBuffer_full );

	return 0;
}

unsigned char play_file() {
	UINT read = 0;
	unsigned short read_size;
	unsigned int blocks_written = 0, max_blocks = DCI_BUFFER_PAGES;

	while( wav_num_channels == 2 && !transitioning_to_next_file && !wav_dummy && !wav_data_align ) {
#ifdef LARGE_READS
		unsigned char NumBlocks = 6;
		signed short* dest = DCI_Prepare_Write_2_Channel_Size(&NumBlocks);
		read_size = 512 * NumBlocks;
		if( f_read(&cur_file, ((BYTE*)dest) + wav_data_align, read_size, &read) != FR_OK ) {
			DCI_Pause(true);
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( read > wav_bytes_remaining )
			read = wav_bytes_remaining;
		wav_bytes_remaining -= read;

        if( CrossFeed_Enabled && g_current_sample_rate <= 48000 ) {
          if( wav_bytes_per_sample == 4 )
            CrossFeed16( (signed short*)(((BYTE*)dest) + wav_data_align), read>>2);
           else
            CrossFeed24( (signed short*)(((BYTE*)dest) + wav_data_align), read/6);
        }

		if( read != read_size || wav_bytes_remaining == 0 ) {
			memset(((BYTE*)dest)+read, 0, read_size-read);
			transitioning_to_next_file = DCI_BUFFER_PAGES;
		}
		DCI_Commit_Write_2_Channel_Size(NumBlocks);
		blocks_written += NumBlocks;
		if( DCIBuffer_full || blocks_written >= max_blocks )
			return 0;
#else
		signed short* dest = DCI_Prepare_Write_2_Channel();
		read_size = 512;
		if( wav_data_align ) {
			if( wav_data_align + read_size >= 512 )
				read_size = 512 - wav_data_align;
			memset(dest, 0, wav_data_align);
		}
		if( f_read(&cur_file, ((BYTE*)dest) + wav_data_align, read_size, &read) != FR_OK ) {
			DCI_Pause(true);
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( read > wav_bytes_remaining )
			read = wav_bytes_remaining;
		wav_bytes_remaining -= read;

        if( CrossFeed_Enabled && g_current_sample_rate <= 48000 ) {
          if( wav_bytes_per_sample == 4 )
            CrossFeed16( (signed short*)(((BYTE*)dest) + wav_data_align), read>>2);
           else
            CrossFeed24( (signed short*)(((BYTE*)dest) + wav_data_align), read/6);
        }

		if( wav_data_align ) {
			wav_data_align += read;
			if( wav_data_align >= 512 )
				wav_data_align = 0;
		}
		if( read != read_size || wav_bytes_remaining == 0 ) {
			memset(((BYTE*)dest)+read, 0, read_size-read);
			transitioning_to_next_file = DCI_BUFFER_PAGES;
		}
		DCI_Commit_Write_2_Channel();
		blocks_written += 1;
		if( DCIBuffer_full || blocks_written >= max_blocks )
			return 0;
#endif
	}

    return play_file_indirect();
}

unsigned char Playback_Process() {
	if( playback_state == playing ) {
		if( play_file() )
			return last_playback_error;
	
		if( transitioning_to_next_file && --transitioning_to_next_file == 0 ) {
			if( play_just_one ) {
				play_just_one = 0;
				if( Playback_Stop() )
					return last_playback_error;
				return Playback_Go_To_First_file();
			} else {
				return Playback_Next_Track_int(0, !dont_repeat_all);
			}
		}
	}
	return last_playback_error;
}


unsigned char Playback_Get_Folder_Level() {
  return cur_folder_level;
}

unsigned char Playback_Next_Folder() {
	if( cur_folder_level > 0 ) {
		Playback_Up_Folder();
		return goto_next_wav_file(true);
	} else {
		return Playback_Next_Track(true);
	}
}

unsigned char Playback_Prev_Folder() {
	if( cur_folder_level > 0 ) {
		Playback_Up_Folder();
		return goto_prev_wav_file(true);
	} else {
		return Playback_Prev_Track(true);
	}
}

unsigned char Playback_Up_Folder() {
	if( cur_folder_level == 0 ) {
#ifdef __DEBUG
		playback_error();
#endif
		playback_state = error;
		last_playback_error = no_wav_files;
	} else {
		char* bs = strrchr(cur_path, '\\');
		char oldchar = '\0';
		if( bs == cur_path+2 ) {
			oldchar = bs[1];
			bs[1] = '\0';
		} else {
			bs[0] = '\0';
		}

		if( f_opendir(&cur_pos, cur_path) != FR_OK ) {
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = open_root_dir_failed;
			return last_playback_error;
		}
		if( oldchar )
			bs[1] = oldchar;
		do {
			if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
				if( oldchar )
					bs[1] = '\0';
#ifdef __DEBUG
				playback_error();
#endif
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			if( (cur_file_info.fattrib&AM_DIR) && !strcmp(cur_file_info.fname, bs+1) )
				break;
		} while( cur_pos.sect != 0 );
		if( oldchar )
			bs[1] = '\0';

		current_playback_order = directory;
        --cur_folder_level;
	}
	return last_playback_error;
}

unsigned char Playback_Get_Current_File(char* buf, unsigned int buflen) {
  if( playback_state != reset && playback_state != error ) {
    const char* src = cur_path;
    while( buflen > 1 && *src ) {
      *buf++ = *src++;
      --buflen;
    }
    if( cur_path[3] != '\0' ) {
      *buf++ = '\\';
      --buflen;
    }
    src = (const char*)cur_file_info.fname;
    while( buflen > 1 && *src ) {
      *buf++ = *src++;
      --buflen;
    }
    *buf = '\0';
    return ok;
  }
  return error;
}

unsigned char Playback_Set_Current_File(const char* buf, unsigned char search_long_names) {
  const char* file = strrchr(buf, '\\');
  int file_index;
  cur_folder_level = 0;
  if( file ) {
    unsigned int len = file - buf;
    if( len > 3 ) {
      if( len > sizeof(cur_path)-1 )
        len = sizeof(cur_path)-1;
      strncpy(cur_path, buf, len);
      cur_path[len] = '\0';
      if( len > 3 ) {
        const char* src = cur_path + 3;
        ++cur_folder_level;
        while( *src ) {
          if( *src == '\\' )
            ++cur_folder_level;
          ++src;
        }
      }
    } else {
      strncpy(cur_path, buf, len+1);
      cur_path[len+1] = '\0';
    }
    if( current_playback_order != directory )
      update_playback_order(0);
    if( (file_index = goto_file_name(file+1, search_long_names)) == -1 || !open_current_file() || do_open_wav() != ok )
      return last_playback_error;
    if( current_playback_order != directory )
	  order_pos = find_file_index(file_index);
    return ok;
  } else {
    return error;
  }
}

unsigned char Playback_Get_Current_Position(unsigned long* pos) {
  if( playback_state != reset && playback_state != error ) {
    *pos = wav_data_size - wav_bytes_remaining;
    return ok;
  }
  return error;
}

unsigned char Playback_Set_Current_Position(unsigned long pos) {
  unsigned long cur_pos = wav_data_size - wav_bytes_remaining;
  if( pos >= cur_pos ) {
    unsigned long seek_distance = pos - cur_pos;
	seek_distance = (seek_distance + 256) & ~511; // round to nearest sector
	if( seek_distance && wav_bytes_remaining > seek_distance ) {
		wav_bytes_remaining -= seek_distance;
		if( !wav_dummy && f_lseek(&cur_file, (wav_data_start) + wav_data_size - wav_bytes_remaining) != FR_OK ) {
			DCI_Pause(true);
#ifdef __DEBUG
			playback_error();
#endif
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
	}
	return ok;
  } else {
    return error;
  }
}
